package com.agilex.healthcare.directscheduling.dataservice;

import com.agilex.healthcare.directscheduling.datalayer.SchedulingDataLayer;
import com.agilex.healthcare.directscheduling.domain.AppointmentSlotFilter;
import com.agilex.healthcare.directscheduling.domain.AppointmentTimeSlot;
import com.agilex.healthcare.directscheduling.domain.AppointmentTimeSlots;
import com.agilex.healthcare.directscheduling.domain.BookAppointment;
import com.agilex.healthcare.directscheduling.domain.BookedAppointment;
import com.agilex.healthcare.directscheduling.domain.BookedAppointmentCollection;
import com.agilex.healthcare.directscheduling.domain.BookedAppointments;
import com.agilex.healthcare.directscheduling.domain.CancelAppointment;
import com.agilex.healthcare.directscheduling.domain.CancelAppointmentEmail;
import com.agilex.healthcare.directscheduling.domain.CancelReasons;
import com.agilex.healthcare.directscheduling.domain.SchedulingClinic;
import com.agilex.healthcare.directscheduling.domain.VARBookedAppointmentCollections;
import com.agilex.healthcare.directscheduling.linkbuilder.DSCancelAppointmentLinkBuilder;
import com.agilex.healthcare.directscheduling.utils.DateHelper;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifiers;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilter;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterFactory;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterQueryParameters;
import com.agilex.healthcare.mobilehealthplatform.utils.QueryParameters;
import gov.va.vamf.scheduling.communitycare.domain.BookedCCAppointment;
import gov.va.vamf.scheduling.direct.datalayer.clinic.ClinicDataService;
import gov.va.vamf.scheduling.direct.datalayer.user.VarPatientCorrelationGateway;
import gov.va.vamf.scheduling.direct.domain.CdwClinic;
import gov.va.vamf.scheduling.direct.domain.CustomFriendlyText;
import gov.va.vamf.videoconnect.common.domain.Appointment;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.context.annotation.Primary;
import org.springframework.context.annotation.Scope;
import org.springframework.scheduling.SchedulingTaskExecutor;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.UriInfo;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;


@Primary
@Service
@Scope(BeanDefinition.SCOPE_SINGLETON)
public class DSDataService {

    @Resource
    private SchedulingDataLayer schedulingDataLayer;

    @Resource
    protected VarPatientCorrelationGateway patientCorrelationGateway;

    @Resource
    private ClinicDataService clinicDataService;

    @Resource
    private SchedulingTaskExecutor executor;

    private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(DSDataService.class);

    private static final String CANCEL_CODE = "PC";
    private static final String GENERIC_BOOKING_ERROR = "An error occurred while booking the appointment.";

    public DSDataService() {
    }

    private AppointmentSlotFilter getAvailableAppointmentSlotsCriteria(UriInfo uriInfo, String siteCode, String clinicId) {
        DateFilter dateFilter = DateFilterFactory.createFilterFromUri(uriInfo);

        AppointmentSlotFilter appointmentSlotFilter = new AppointmentSlotFilter();
        appointmentSlotFilter.setClinicId(clinicId);
        appointmentSlotFilter.setSiteCode(siteCode);
        appointmentSlotFilter.setStartDate(dateFilter.getStartDate());
        appointmentSlotFilter.setEndDate(dateFilter.getEndDate());

        return appointmentSlotFilter;
    }

    private AppointmentSlotFilter getBookedAppointmentFilterCriteria(UriInfo uriInfo, String siteCode) {
        QueryParameters queryParameters = new QueryParameters(uriInfo.getRequestUri().getQuery());
        DateFilterQueryParameters parameters = new DateFilterQueryParameters(queryParameters);

        AppointmentSlotFilter appointmentSlotFilter = new AppointmentSlotFilter();
        appointmentSlotFilter.setSiteCode(siteCode);
        appointmentSlotFilter.setStartDate(DateHelper.determineStartDate(null, parameters));
        return appointmentSlotFilter;
    }

    public Response cancelAppointmentForPatient(CancelAppointment cancelAppointment, PatientIdentifier patientIdentifier, String siteCode, HttpServletRequest request) {
        cancelAppointment.setCancelCode(CANCEL_CODE);

        PatientIdentifier icnPatientIdentifier = getIcnIdentifiers(patientIdentifier);
        if (icnPatientIdentifier == null) {
            return (null);
        }

        cancelAppointment.setPatientIdentifier(icnPatientIdentifier);
        CancelAppointment canceledAppointment = schedulingDataLayer.cancelAppointmentForPatient(cancelAppointment, siteCode);
        if (isAppointmentCanceledSuccessfully(cancelAppointment, canceledAppointment)) {
            request.getSession().setAttribute("cancel-appointment", (CancelAppointmentEmail) canceledAppointment);
            return Response.ok().build();
        }

        return Response.status(Response.Status.INTERNAL_SERVER_ERROR).entity(GENERIC_BOOKING_ERROR).build();
    }

    public CancelReasons getCancelReasonList(PatientIdentifier patientIdentifier, String siteCode) {
        PatientIdentifier icnPatientIdentifier = getIcnIdentifiers(patientIdentifier);
        if (icnPatientIdentifier == null) {
            return (null);
        }

        CancelReasons cancelReasons = schedulingDataLayer.getCancelReasonList(icnPatientIdentifier, siteCode);
        return cancelReasons;
    }

    public VARBookedAppointmentCollections getPatientBookedAppointments(UriInfo uriInfo, PatientIdentifier patientIdentifier,
                                                                        String siteCode, String clinicIds) {
        VARBookedAppointmentCollections bookedAppointmentCollections = new VARBookedAppointmentCollections();

        PatientIdentifier icnPatientIdentifier = getIcnIdentifiers(patientIdentifier);
        if (icnPatientIdentifier == null) {
            return (null);
        }

        Collection<BookedAppointmentCollection> bookedAppointmentCollectionContainer = bookedAppointmentCollections.getBookedAppointmentCollections();

        if (icnPatientIdentifier != null) {
            AppointmentSlotFilter appointmentSlotFilter = getBookedAppointmentFilterForPatient(uriInfo, siteCode, icnPatientIdentifier.getUniqueId());

            appointmentSlotFilter.setPatientIdentifier(icnPatientIdentifier);
            BookedAppointments bookedAppointments = schedulingDataLayer.getPatientBookedAppointments(appointmentSlotFilter);

            DSCancelAppointmentLinkBuilder linkBuilder = new DSCancelAppointmentLinkBuilder(uriInfo.getBaseUri());
            linkBuilder.fillLinks(bookedAppointments, patientIdentifier, siteCode);

            Collection<String> clinicIdCollection = null;

            if (clinicIds != null) {
                clinicIdCollection = Arrays.asList(clinicIds.split(","));
            }

            // obtain all child facilities (institutions)/clinics map for given site code and clinic IENs
            final Set<String> clinicIens = new HashSet<String>(bookedAppointments.size());
            for (final BookedAppointment appointment : bookedAppointments) {
                // when clinic ID's is null, get all booked appointment clinic ID's.
                // otherwise, match clinic ID's
                if (clinicIds == null ||
                        (clinicIdCollection != null && clinicIdCollection.contains(appointment.getClinic().getId()))) {

                    clinicIens.add(appointment.getClinic().getId());
                }
            }
            List<CdwClinic> displayToPatientClinics = clinicDataService.retrieveDisplayToPatientClinics(siteCode, clinicIens);


            Map<String, CdwClinic> childMap = getChildFacilityToClinicMap(displayToPatientClinics);

            Collection<BookedAppointment> filteredAppointments =
                filterBookedAppointmentsByClinics(displayToPatientClinics, bookedAppointments.getBookedAppointment(), childMap);

            if (filteredAppointments != null) {
                // Now let's add the primary care appointments to a new BookedAppointmentCollection and set the name.
                BookedAppointmentCollection primaryCareBookedAppointmentCollection = new BookedAppointmentCollection();
                primaryCareBookedAppointmentCollection.setCollectionName("bookedAppointments");
                primaryCareBookedAppointmentCollection.setBookedAppointments(filteredAppointments);

                // Add the new primary care BookedAppointmentCollection to the container holding all BookedAppointmentCollections.
                bookedAppointmentCollectionContainer.add(primaryCareBookedAppointmentCollection);
            }
        }

        return (bookedAppointmentCollections);
    }

    public VARBookedAppointmentCollections getPatientCCBookedAppointments(List<BookedCCAppointment> bookedCCAppointmentList) {
        VARBookedAppointmentCollections bookedAppointmentCollections = new VARBookedAppointmentCollections();
        Collection<BookedAppointmentCollection> bookedAppointmentCollectionContainer = bookedAppointmentCollections.getBookedAppointmentCollections();
        BookedAppointmentCollection bookedCCAppointmentCollection = new BookedAppointmentCollection();
        bookedCCAppointmentCollection.setCollectionName("bookedCCAppointments");
        bookedCCAppointmentCollection.setBookedCCAppointments(bookedCCAppointmentList);

        // Add the new primary care BookedAppointmentCollection to the container holding all BookedAppointmentCollections.
        bookedAppointmentCollectionContainer.add(bookedCCAppointmentCollection);
        return bookedAppointmentCollections;

    }

    private Collection<BookedAppointment> filterBookedAppointmentsByClinics(List<CdwClinic> displayToPatientClinics,
                                                                            Collection<BookedAppointment> bookedAppointments, Map<String, CdwClinic> childMap) {
        Map<String, String> displayToPatientClinicsMap = new HashMap<String, String>();
        for (CdwClinic clinic : displayToPatientClinics) {
            String id = clinic.getClinicId();
            displayToPatientClinicsMap.put(id, id);
        }

        Collection<BookedAppointment> filteredAppointments = new ArrayList<BookedAppointment>();
        for (BookedAppointment appointment : bookedAppointments) {
            CdwClinic cdwClinic = null;
            final SchedulingClinic appointmentClinic = appointment.getClinic();
            String appointmentClinicId = appointmentClinic.getId();
            if (appointmentClinicId.equals(displayToPatientClinicsMap.get(appointmentClinicId))) {
                if(childMap.containsKey(appointmentClinicId)) {
                    cdwClinic = childMap.get(appointmentClinicId);

                    appointmentClinic.setFacilityCode(cdwClinic.getInstitutionCode());
                    // Yes, unfortunately, the clinic's "friendly name" is the institution's name, and its
                    // "clinic friendly name" is the actual friendly name of the clinic.
                    appointmentClinic.setFriendlyName(cdwClinic.getInstitutionName());
                    appointmentClinic.setClinicFriendlyName(cdwClinic.getClinicFriendlyLocationName());
                }
                filteredAppointments.add(appointment);
            }
        }

        return filteredAppointments;
    }

    public Response bookNewAppointmentForPatient(Appointment appointment,
                                                 PatientIdentifier patientIdentifier, String siteCode, HttpServletRequest request) {
        PatientIdentifier icnPatientIdentifier = getIcnIdentifiers(patientIdentifier);
        if (icnPatientIdentifier == null) {
            return (null);
        }

        BookAppointment bookAppointment = new BookAppointment(appointment);
        bookAppointment.setPatientIdentifier(icnPatientIdentifier);
        BookAppointment newBookedAppointment;

        try {
            newBookedAppointment = schedulingDataLayer.bookPatientAppointment(bookAppointment, siteCode);
        } catch (Exception e) {
            return Response.status(Response.Status.BAD_REQUEST).entity("Error: " + e.toString()).type(MediaType.APPLICATION_JSON).build();
        }
        if (newBookedAppointment.getError() != null && newBookedAppointment.getError().length() > 0) {
            return Response.status(Response.Status.BAD_REQUEST).entity("Error: " + newBookedAppointment.getError()).type(MediaType.APPLICATION_JSON).build();
        }
        if (isAppointmentSavedCorrectly(bookAppointment, newBookedAppointment)) {

            return Response.ok(appointment).build();
        }
        return Response.status(Response.Status.BAD_REQUEST).entity(GENERIC_BOOKING_ERROR).type(MediaType.APPLICATION_JSON).build();
    }

    private AppointmentSlotFilter getBookedAppointmentFilterForPatient(UriInfo uriInfo, String siteCode, String patientId) {
        AppointmentSlotFilter appointmentSlotFilter = getBookedAppointmentFilterCriteria(uriInfo, siteCode);
        appointmentSlotFilter.setPatientId(patientId);
        return appointmentSlotFilter;
    }

    private boolean isAppointmentSavedCorrectly(BookAppointment requestedAppointment, BookAppointment savedAppointment) {
        return (requestedAppointment.getDateTime().equals(savedAppointment.getDateTime())
            && requestedAppointment.getClinicId().equals(savedAppointment.getClinicId())
            && requestedAppointment.getPurpose().equals(savedAppointment.getPurpose()));
    }

    private boolean isAppointmentCanceledSuccessfully(CancelAppointment originalObject, CancelAppointment returnObject) {
        return (originalObject.getClinicId().equals(returnObject.getClinicId())
            && originalObject.getAppointmentTime().equals(returnObject.getAppointmentTime())
            && originalObject.getCancelCode().equals(returnObject.getCancelCode())
            && originalObject.getCancelReason().equals(returnObject.getCancelReason()));
    }

    public Collection<AppointmentTimeSlots> getFilteredAppointmentSlots(UriInfo uriInfo, PatientIdentifier patientIdentifier, String siteCode, String clinicIds) throws InterruptedException, ExecutionException {
        Collection<AppointmentTimeSlots> collectionOfApointmentTimeSlots = null;

        PatientIdentifier icnPatientIdentifier = getIcnIdentifiers(patientIdentifier);
        if (icnPatientIdentifier == null) {
            return (null);
        }

        if (clinicIds != null && clinicIds.length() >= 1) {

            List<String> clinicIdList = new ArrayList<String>(Arrays.asList(StringUtils.split(clinicIds, ",")));

            collectionOfApointmentTimeSlots = new ArrayList<AppointmentTimeSlots>();

            class ScheduleFetcher implements Callable<AppointmentTimeSlots> {
                String clinicId;
                PatientIdentifier icnPatientIdentifier;
                UriInfo uriInfo;
                String siteCode;

                public ScheduleFetcher(String clinicId, PatientIdentifier icnPatientIdentifier, UriInfo uriInfo, String siteCode){
                    ScheduleFetcher.this.clinicId = clinicId;
                    ScheduleFetcher.this.icnPatientIdentifier = icnPatientIdentifier;
                    ScheduleFetcher.this.uriInfo = uriInfo;
                    ScheduleFetcher.this.siteCode = siteCode;
                }
                @Override
                public AppointmentTimeSlots call() throws Exception {
                    AppointmentSlotFilter appointmentSlotFilter = getAvailableAppointmentSlotsCriteria(uriInfo, siteCode, clinicId);
                    appointmentSlotFilter.setPatientIdentifier(icnPatientIdentifier);
                    AppointmentTimeSlots appointmentTimeSlots = schedulingDataLayer.getClinicSchedulingDetail(appointmentSlotFilter);
                    populateWithCdwStopCodes(appointmentTimeSlots, clinicId, fetchDisplayToPatientClinicsMap(siteCode));

                    Collection<AppointmentTimeSlot> appointmentTimeSlotsByClinic = filterAppointmentTimeSlotsByCriteria(appointmentTimeSlots.getAppointmentTimeSlot());
                    appointmentTimeSlots.removeAll(appointmentTimeSlots.getAppointmentTimeSlot());

                    if (appointmentTimeSlotsByClinic != null && appointmentTimeSlotsByClinic.size() > 0) {
                        appointmentTimeSlots.addAll(filterAppointmentActiveAppointments(appointmentTimeSlotsByClinic));
                    }

                    return appointmentTimeSlots;
                }
            }

            List<Future<AppointmentTimeSlots>> futures = new LinkedList<Future<AppointmentTimeSlots>>();

            for( String clinicId : clinicIdList ){
                final Future<AppointmentTimeSlots> future = executor.submit(new ScheduleFetcher(clinicId, icnPatientIdentifier, uriInfo, siteCode));
                futures.add(future);
            }

            try {
                for(Future<AppointmentTimeSlots> future : futures){
                    collectionOfApointmentTimeSlots.add(future.get());
                }
            } catch (InterruptedException e) {
                e.printStackTrace();
                throw e;
            } catch (ExecutionException e) {
                e.printStackTrace();
                throw e;
            }
        }

        return (collectionOfApointmentTimeSlots);
    }

    private void populateWithCdwStopCodes(AppointmentTimeSlots appointmentTimeSlots, String clinicId,
                                          Map<String, CdwClinic> displayToPatientClinicsMap) {

        CdwClinic cdwClinic = displayToPatientClinicsMap.get(clinicId);
        if (cdwClinic != null) {
            appointmentTimeSlots.setPrimaryStopCode(cdwClinic.getPrimaryStopCode());
            appointmentTimeSlots.setSecondaryStopCode(cdwClinic.getSecondaryStopCode());
        }
    }

    private Map<String, CdwClinic> fetchDisplayToPatientClinicsMap(String siteCode) {
        List<CdwClinic> displayToPatientClinics = clinicDataService.retrieveDisplayToPatientClinics(siteCode);

        Map<String, CdwClinic> map = new HashMap<String, CdwClinic>();
        for (CdwClinic clinic : displayToPatientClinics) {
            String clinicId = clinic.getClinicId();
            map.put(clinicId, clinic);
        }

        return map;
    }

    protected List<String> getIcnIdentifiers(PatientIdentifiers correspondIdentifiers) {
        List<String> foundIcnIdentifiers = new ArrayList<String>();
        for (PatientIdentifier correspondIdentifier : correspondIdentifiers) {
            if ("ICN".equalsIgnoreCase(correspondIdentifier.getAssigningAuthority())) {
                foundIcnIdentifiers.add(correspondIdentifier.getUniqueId());
            }
        }
        return foundIcnIdentifiers;
    }

    public PatientIdentifier getIcnIdentifiers(PatientIdentifier patientIdentifier) {
        PatientIdentifiers correspondIdentifiers = patientCorrelationGateway.getCorrespondIdentifiers(patientIdentifier);
        List<String> icnIdentifiers = getIcnIdentifiers(correspondIdentifiers);
        return ((icnIdentifiers == null || icnIdentifiers.isEmpty()) ? null : new PatientIdentifier("ICN", stripHash(icnIdentifiers.get(0))));
    }

    private AppointmentTimeSlots filterAppointmentActiveAppointments(Collection<AppointmentTimeSlot> appointmentTimeSlotsByClinics) {
        AppointmentTimeSlots tempAppointmentTimeSlots = new AppointmentTimeSlots();

        for (AppointmentTimeSlot appointmentTimeSlot : appointmentTimeSlotsByClinics) {
            if (appointmentTimeSlot.isAvailability()) {
                tempAppointmentTimeSlots.add(appointmentTimeSlot);
            }
        }

        return tempAppointmentTimeSlots;
    }

    private Collection<AppointmentTimeSlot> filterAppointmentTimeSlotsByCriteria(Collection<AppointmentTimeSlot> appointmentTimeSlotsByClinic) {

        Collection<AppointmentTimeSlot> appointmentTimeSlots = new ArrayList<AppointmentTimeSlot>();

         for (AppointmentTimeSlot appointmentTimeSlot : appointmentTimeSlotsByClinic) {
            if (!appointmentTimeSlot.isAvailability()) {
                continue;
            }
            if (!isAppointmentNomenclatureValid(appointmentTimeSlot)) {
                continue;
            }

            appointmentTimeSlots.add(appointmentTimeSlot);
        }

        return appointmentTimeSlots;
    }

    private boolean isAppointmentNomenclatureValid(AppointmentTimeSlot appointmentTimeSlot) {
        if (appointmentTimeSlot != null) {
            String bookingStatus = appointmentTimeSlot.getBookingStatus();
            char ch = bookingStatus.length() > 0 ? bookingStatus.charAt(0) : '0';
            if ((ch >= '1') && (ch <= '9') || (ch >= 106) && (ch <= 122)) {
                return (true);
            }
            return (false);
        }
        return (false);
    }

    private String stripHash(String originalIcn) {

        int index = originalIcn.toLowerCase().indexOf("v");
        if (index <= 0)
            return originalIcn;

        return originalIcn.substring(0, index);
    }

    private Map<String, CdwClinic> getChildFacilityToClinicMap(List<CdwClinic> childFacilityToClinicList) {
        // return back if child facilities / clinics from CDW do not exist
        Map<String, CdwClinic> childFacilityToClinicMap = new HashMap<>();
        if(childFacilityToClinicList.isEmpty()) {
            return childFacilityToClinicMap;
        }

        // retrieve the child facility custom friendly name from Mongo
        Map<String, CustomFriendlyText> childFacilityCustomFriendlyTextMap = new HashMap<String, CustomFriendlyText>();
//        final CustomFriendlyTexts customFriendlyTexts = clinicDataService.retrieveChildFacilityCustomFriendlyTexts();
        for(CdwClinic clinic : childFacilityToClinicList) {
            CustomFriendlyText friendlyText = clinicDataService.retrieveChildFacilityCustomFriendlyTexts(clinic.getInstitutionCode());
            //if the clinic has a friendly Name add it to the map
            if (friendlyText != null) {
                childFacilityCustomFriendlyTextMap.put(clinic.getInstitutionCode(), friendlyText);
            }
        }

        // update the child facility name with its custom friendly text if found in childFacilityCustomFriendlyTextMap
        CustomFriendlyText tmpCustomFriendlyText;
        String childFacilityCode, clinicId;
        for (CdwClinic theClinic : childFacilityToClinicList) {
            clinicId = theClinic.getClinicId();
            childFacilityCode = theClinic.getInstitutionCode();
            if (childFacilityCustomFriendlyTextMap.containsKey(childFacilityCode)) {
                tmpCustomFriendlyText = childFacilityCustomFriendlyTextMap.get(childFacilityCode);
                theClinic.setInstitutionName(tmpCustomFriendlyText.getFriendlyText());
            }
            childFacilityToClinicMap.put(clinicId, theClinic);
        }

        return childFacilityToClinicMap;
    }
}